#!/bin/bash

# ====== 配置参数 ======
HOME_PATH={vectorPath}
PGVECTOR_VERSION="0.8.1"
PGVECTOR_ARCHIVE="pgvector-${PGVECTOR_VERSION}.tar.gz"

# PostgreSQL 目录配置
PG_DIR="/usr/pgsql-13"
PG_CONFIG="$PG_DIR/bin/pg_config"
PG_LIB_DIR="$PG_DIR/lib"
PG_SHARE_DIR="$PG_DIR/share/extension"

# ====== 检查是否已安装 ======
echo "检查是否已编译安装 pgvector..."

# 搜索 vector.so 和 vector.control
VECTOR_SO="$PG_LIB_DIR/vector.so"
VECTOR_CONTROL="$PG_SHARE_DIR/vector.control"

if [ -f "$VECTOR_SO" ] && [ -f "$VECTOR_CONTROL" ]; then
    echo "pgvector 已编译安装（找到 vector.so 和 vector.control），跳过安装步骤。"
    echo "VECTOR_SO: $VECTOR_SO"
    echo "VECTOR_CONTROL: $VECTOR_CONTROL"
    exit 0
else
    echo "未找到完整的 pgvector 安装，继续安装过程..."
    echo "查找结果:"
    echo "  $VECTOR_SO: $( [ -f "$VECTOR_SO" ] && echo "存在" || echo "不存在" )"
    echo "  $VECTOR_CONTROL: $( [ -f "$VECTOR_CONTROL" ] && echo "存在" || echo "不存在" )"
fi

# ====== 验证 pg_config ======
echo "验证 PostgreSQL 配置..."
if [ ! -f "$PG_CONFIG" ]; then
    echo "错误: 未找到 pg_config ($PG_CONFIG)"
    exit 1
fi

echo "使用 pg_config: $PG_CONFIG"
$PG_CONFIG --version

# ====== 解压源码 ======
echo "解压 pgvector 源码..."
mkdir -p "$HOME_PATH/tmps"
cd "$HOME_PATH"

if [ ! -f "$PGVECTOR_ARCHIVE" ]; then
    echo "错误: 未找到源码包 $PGVECTOR_ARCHIVE"
    exit 1
fi

tar -xzf "$PGVECTOR_ARCHIVE" -C "$HOME_PATH/tmps"

# 查找解压后的目录
PGVECTOR_SRC_DIR=$(find "$HOME_PATH/tmps" -maxdepth 1 -type d -name "pgvector*" | head -n 1)

if [ -z "$PGVECTOR_SRC_DIR" ]; then
    echo "错误: 解压失败，未找到 pgvector 源码目录"
    exit 1
fi

echo "进入源码目录: $PGVECTOR_SRC_DIR"
cd "$PGVECTOR_SRC_DIR"

# ====== 编译安装 ======
echo "开始编译 pgvector..."

# 设置环境变量
export PG_CONFIG="$PG_CONFIG"
export PATH="$PG_DIR/bin:$PATH"

# 清理之前的编译结果（如果有）
make clean 2>/dev/null || true

# 编译
echo "执行 make..."
make USE_PGXS=1 PG_CONFIG="$PG_CONFIG" || {
    echo "错误: 编译失败"
    exit 1
}

# 安装
echo "执行 make install..."
make USE_PGXS=1 PG_CONFIG="$PG_CONFIG" install || {
    echo "错误: 安装失败"
    exit 1
}

# ====== 验证安装 ======
echo "验证安装结果..."

if [ -f "$VECTOR_SO" ]; then
    echo "✓ vector.so 已安装: $VECTOR_SO"
else
    echo "✗ 错误: vector.so 未找到"
    exit 1
fi

if [ -f "$VECTOR_CONTROL" ]; then
    echo "✓ vector.control 已安装: $VECTOR_CONTROL"
else
    echo "✗ 错误: vector.control 未找到"
    exit 1
fi

# 检查 SQL 文件
VECTOR_SQL="$PG_SHARE_DIR/vector--${PGVECTOR_VERSION}.sql"
if [ -f "$VECTOR_SQL" ]; then
    echo "✓ SQL 文件已安装: $VECTOR_SQL"
else
    # 尝试查找其他版本的 SQL 文件
    VECTOR_SQL_FOUND=$(find "$PG_SHARE_DIR" -name "vector--*.sql" | head -n 1)
    if [ -n "$VECTOR_SQL_FOUND" ]; then
        echo "✓ SQL 文件已安装: $VECTOR_SQL_FOUND"
    else
        echo "⚠ 警告: 未找到 SQL 文件"
    fi
fi


# ====== 清理临时文件 ======
echo "清理临时文件..."
rm -rf "$HOME_PATH/tmps"

# ====== 完成 ======
echo ""
echo "=========================================="
echo "pgvector 安装完成！"
echo "=========================================="
echo ""
echo "安装信息:"
echo "  版本: $PGVECTOR_VERSION"
echo "  SO 文件: $VECTOR_SO"
echo "  Control 文件: $VECTOR_CONTROL"
echo ""
echo "下一步操作:"
echo "1. 连接到 PostgreSQL 数据库:"
echo "   psql -U postgres -d your_database"
echo ""
echo "2. 在数据库中启用 vector 扩展:"
echo "   CREATE EXTENSION vector;"
echo ""
echo "3. 验证扩展安装:"
echo "   SELECT * FROM pg_extension WHERE extname = 'vector';"
echo ""
echo "4. 创建向量表示例:"
echo "   CREATE TABLE items (id bigserial PRIMARY KEY, embedding vector(1536));"
echo ""
echo "5. 创建向量索引:"
echo "   CREATE INDEX ON items USING ivfflat (embedding vector_l2_ops) WITH (lists = 100);"
echo "   -- 或使用 HNSW 索引 (需要 0.5.0+)"
echo "   CREATE INDEX ON items USING hnsw (embedding vector_l2_ops);"
echo ""
echo "更多信息: https://github.com/pgvector/pgvector"
echo "=========================================="

exit 0