#!/bin/bash

# ====== 配置参数 ======
HOME_PATH={zhparserPath}
ZHPARSER_VERSION="2.3"
ZHPARSER_ARCHIVE="zhparser-${ZHPARSER_VERSION}.tar.gz"
SCWS_VERSION="1.2.3"
SCWS_ARCHIVE="scws-${SCWS_VERSION}.tar.gz"

# PostgreSQL 目录配置
PG_DIR="/usr/pgsql-13"
PG_CONFIG="$PG_DIR/bin/pg_config"
PG_LIB_DIR="$PG_DIR/lib"
PG_SHARE_DIR="$PG_DIR/share/extension"
ZHPARSER_SO="$PG_LIB_DIR/zhparser.so"
ZHPARSER_CONTROL="$PG_SHARE_DIR/zhparser.control"

# SCWS 安装目录
SCWS_PREFIX="/usr/local"

export PATH="/usr/local/bin:$PATH"
export LD_LIBRARY_PATH="/usr/local/lib:$LD_LIBRARY_PATH"

# ====== 检查是否已安装 ======
echo "检查是否已编译安装 SCWS 和 zhparser..."

# 检查 SCWS 是否已安装
if [ -f "/usr/local/bin/scws" ] && [ -f "/usr/local/lib/libscws.so" ]; then
    echo "✓ SCWS已安装，跳过安装步骤"
    SCWS_INSTALLED=true
else
    echo "SCWS未安装，开始安装..."
    SCWS_INSTALLED=false
fi

# 检查 zhparser 是否已安装
if [ -f "$ZHPARSER_SO" ] && [ -f "$ZHPARSER_CONTROL" ]; then
    echo "zhparser 已编译安装（找到 zhparser.so 和 zhparser.control），跳过安装步骤。"
    echo "ZHPARSER_SO: $ZHPARSER_SO"
    echo "ZHPARSER_CONTROL: $ZHPARSER_CONTROL"
    
    if [ "$SCWS_INSTALLED" = true ]; then
        echo "SCWS 和 zhparser 均已安装，无需继续执行。"
        exit 0
    fi
else
    echo "未找到完整的 zhparser 安装，继续安装过程..."
    echo "查找结果:"
    echo "  $ZHPARSER_SO: $( [ -f "$ZHPARSER_SO" ] && echo "存在" || echo "不存在" )"
    echo "  $ZHPARSER_CONTROL: $( [ -f "$ZHPARSER_CONTROL" ] && echo "存在" || echo "不存在" )"
fi

# ====== 安装 SCWS ======
if [ "$SCWS_INSTALLED" = false ]; then
    echo "开始安装 SCWS..."
    
    # 检查 SCWS 源码包
    if [ ! -f "$SCWS_ARCHIVE" ]; then
        echo "错误: 未找到 SCWS 源码包 $SCWS_ARCHIVE"
        echo "请从以下地址下载 scws-${SCWS_VERSION}.tar.bz2:"
        echo "  http://www.xunsearch.com/scws/download.php"
        exit 1
    fi
    
    # 创建临时目录
    mkdir -p "$HOME_PATH/tmps"
    
    # 解压 SCWS 源码
    echo "解压 SCWS 源码..."
    tar -xzf "$SCWS_ARCHIVE" -C "$HOME_PATH/tmps"
    
    # 查找解压后的 SCWS 目录
    SCWS_SRC_DIR=$(find "$HOME_PATH/tmps" -maxdepth 1 -type d -name "scws*" | head -n 1)
    
    if [ -z "$SCWS_SRC_DIR" ]; then
        echo "错误: SCWS 解压失败，未找到源码目录"
        exit 1
    fi
    
    echo "进入 SCWS 源码目录: $SCWS_SRC_DIR"
    cd "$SCWS_SRC_DIR"
    
    # 配置 SCWS
    echo "配置 SCWS..."
    ./configure --prefix="$SCWS_PREFIX" || {
        echo "错误: SCWS 配置失败"
        echo "可能的原因: 缺少编译工具，请执行: yum install gcc make autoconf"
        exit 1
    }
    
    # 编译 SCWS
    echo "编译 SCWS..."
    make || {
        echo "错误: SCWS 编译失败"
        exit 1
    }
    
    # 安装 SCWS
    echo "安装 SCWS..."
    make install || {
        echo "错误: SCWS 安装失败"
        exit 1
    }
    
    # 更新动态链接库缓存
    ldconfig
    
    # ====== 验证 SCWS 安装 ======
    echo "验证 SCWS 安装..."

    # 更新动态链接库缓存
    sudo ldconfig 2>/dev/null || ldconfig 2>/dev/null || echo "警告: 需要root权限更新ldconfig"

    if command -v scws >/dev/null 2>&1 && ldconfig -p | grep -q libscws; then
        echo "✓ SCWS 安装成功"
        echo "  SCWS 版本: $(scws -v 2>/dev/null || echo "版本信息不可用")"
        echo "  libscws 路径: $(ldconfig -p | grep libscws | head -1)"
    else
        echo "⚠ SCWS 安装验证需要环境变量设置"
        echo "  请执行以下命令或添加到 ~/.bashrc:"
        echo "  export PATH=\"/usr/local/bin:\$PATH\""
        echo "  export LD_LIBRARY_PATH=\"/usr/local/lib:\$LD_LIBRARY_PATH\""
        echo "  source ~/.bashrc"
        
        # 尝试直接验证库文件存在
        if [ -f "/usr/local/lib/libscws.so" ] || [ -f "/usr/local/lib/libscws.so.1.1.0" ]; then
            echo "✓ SCWS 库文件已安装，继续zhparser安装..."
        else
            echo "✗ SCWS 安装失败，库文件未找到"
            exit 1
        fi
    fi
    
    # 返回原始目录
    cd "$HOME_PATH"
fi

# ====== 验证 pg_config ======
echo "验证 PostgreSQL 配置..."
if [ ! -f "$PG_CONFIG" ]; then
    echo "错误: 未找到 pg_config ($PG_CONFIG)"
    exit 1
fi

echo "使用 pg_config: $PG_CONFIG"
$PG_CONFIG --version

# ====== 安装 zhparser ======
echo "开始安装 zhparser..."

# 检查 zhparser 源码包
if [ ! -f "$ZHPARSER_ARCHIVE" ]; then
    echo "错误: 未找到源码包 $ZHPARSER_ARCHIVE"
    echo "请从以下地址下载 zhparser-${ZHPARSER_VERSION}.tar.gz:"
    echo "  https://github.com/amutu/zhparser/archive/refs/tags/V${ZHPARSER_VERSION}.tar.gz"
    exit 1
fi

# 解压 zhparser 源码
echo "解压 zhparser 源码..."
tar -xzf "$ZHPARSER_ARCHIVE" -C "$HOME_PATH/tmps"

# 查找解压后的 zhparser 目录
ZHPARSER_SRC_DIR=$(find "$HOME_PATH/tmps" -maxdepth 1 -type d -name "zhparser*" | head -n 1)

if [ -z "$ZHPARSER_SRC_DIR" ]; then
    echo "错误: zhparser 解压失败，未找到源码目录"
    exit 1
fi

echo "进入 zhparser 源码目录: $ZHPARSER_SRC_DIR"
cd "$ZHPARSER_SRC_DIR"

# ====== 编译安装 zhparser ======
echo "开始编译 zhparser..."

# 设置环境变量
export PG_CONFIG="$PG_CONFIG"
export PATH="$PG_DIR/bin:$PATH"
export SCWS_HOME="$SCWS_PREFIX"

# 清理之前的编译结果（如果有）
make clean 2>/dev/null || true

# 编译 zhparser
echo "执行 make..."
make PG_CONFIG="$PG_CONFIG" SCWS_HOME="$SCWS_PREFIX" || {
    echo "错误: zhparser 编译失败"
    echo ""
    echo "可能的原因:"
    echo "1. SCWS 安装不完整，请检查 SCWS 安装日志"
    echo "2. 缺少 PostgreSQL 开发包"
    echo "3. 环境变量设置问题，请确认 SCWS_HOME 路径正确"
    exit 1
}

# 安装 zhparser
echo "执行 make install..."
make PG_CONFIG="$PG_CONFIG" install || {
    echo "错误: zhparser 安装失败"
    exit 1
}

# ====== 验证安装 ======
echo "验证安装结果..."

if [ -f "$ZHPARSER_SO" ]; then
    echo "zhparser.so 已安装: $ZHPARSER_SO"
    # 显示 SO 文件依赖
    echo "  依赖库检查:"
    ldd "$ZHPARSER_SO" | grep scws && echo "  检测到 scws 依赖" || echo "  警告: 未检测到 scws 依赖"
else
    echo "✗ 错误: zhparser.so 未找到"
    exit 1
fi

if [ -f "$ZHPARSER_CONTROL" ]; then
    echo "zhparser.control 已安装: $ZHPARSER_CONTROL"
else
    echo "错误: zhparser.control 未找到"
    exit 1
fi

# 检查 SQL 文件
ZHPARSER_SQL="$PG_SHARE_DIR/zhparser--${ZHPARSER_VERSION}.sql"
if [ -f "$ZHPARSER_SQL" ]; then
    echo "✓ SQL 文件已安装: $ZHPARSER_SQL"
else
    # 尝试查找其他版本的 SQL 文件
    ZHPARSER_SQL_FOUND=$(find "$PG_SHARE_DIR" -name "zhparser--*.sql" | head -n 1)
    if [ -n "$ZHPARSER_SQL_FOUND" ]; then
        echo "SQL 文件已安装: $ZHPARSER_SQL_FOUND"
    else
        echo "警告: 未找到 SQL 文件"
    fi
fi

# ====== 清理临时文件 ======
echo "清理临时文件..."
rm -rf "$HOME_PATH/tmps"

# ====== 完成 ======
echo ""
echo "=========================================="
echo "SCWS 和 zhparser 安装完成！"
echo "=========================================="
echo ""
echo "安装信息:"
echo "  SCWS 版本: $SCWS_VERSION"
echo "  zhparser 版本: $ZHPARSER_VERSION"
echo "  SCWS 安装路径: $SCWS_PREFIX"
echo "  zhparser SO 文件: $ZHPARSER_SO"

exit 0